window.NaBezkyMap = {
    config: {
        containerId: null,
        mapUrl: null,
        region: null,
        iframe: null,
        cookieName: 'nb_access_token'
    },

    init: function(options) {
        console.log('NaBezkyMap init with options:', options);
        this.config = { ...this.config, ...options };
        this.createIframe();
        
        console.log('Adding message listener');
        window.addEventListener('message', (event) => {
            console.log('Parent received message:', event);
            this.handleMessage(event);
        });
    },

    createIframe: function() {
        console.log('Creating iframe');
        const container = document.getElementById(this.config.containerId);
        if (!container) {
            console.error('Container not found:', this.config.containerId);
            return;
        }

        // Start with base URL
        const iframeUrl = new URL(this.config.mapUrl);
        
        // Add region/ski_center if present
        if (this.config.region && iframeUrl.pathname.includes('groominregion')) {
            iframeUrl.pathname += '/' + this.config.region + '/' + this.config.ski_center;
        }

        // Pass through all URL parameters from parent page
        const currentUrlParams = new URLSearchParams(window.location.search);
        currentUrlParams.forEach((value, key) => {
            iframeUrl.searchParams.set(key, value);
        });

        // Add voucher state if cookie exists
        const existingState = this.getVoucherState();
        if (existingState) {
            const encodedState = btoa(JSON.stringify(existingState));
            iframeUrl.searchParams.set('voucherState', encodedState);
            console.log('Added voucher state to iframe URL');
        }
        
        this.config.iframe = document.createElement('iframe');
        this.config.iframe.src = iframeUrl.toString();
        this.config.iframe.className = 'map-iframe';
        container.appendChild(this.config.iframe);
        console.log('Iframe created with URL:', iframeUrl.toString());
    },
    
    handleMessage: function(event) {
        if (!this.isAllowedOrigin(event.origin)) {
            console.log('Rejected message from unauthorized origin:', event.origin);
            return;
        }
        
        const data = event.data;
        if (data.type === 'voucherValidated') {
            console.log('Processing voucher validation');
            
            const tokenData = {
                voucher: data.voucher,
                email: data.email,
                expires: data.expires,
                timestamp: Date.now()
            };
            
            console.log('Token data:', tokenData);
            const token = btoa(JSON.stringify(tokenData));
            this.setAccessCookie(token);
            
            if (this.config.iframe?.contentWindow) {
                this.config.iframe.contentWindow.postMessage({
                    type: 'cookieSet',
                    state: tokenData
                }, '*');
            }
        }
    },

    setAccessCookie: function(token) {
        try {
            const decodedToken = JSON.parse(atob(token));
            const expiryDate = new Date(decodedToken.expires * 1000);
            const domain = this.getBaseDomain(window.location.hostname);
            
            // Set the cookie
            const cookieString = `${this.config.cookieName}=${token}; ` +
                `expires=${expiryDate.toUTCString()}; ` +
                `path=/; ` +
                `domain=${domain}; ` +
                'SameSite=Lax; ' +
                'Secure';
            
            document.cookie = cookieString;
            
            // Verify the cookie and check for ITP
            const savedCookie = this.getAccessCookie();
            if (savedCookie) {
                // Set up periodic cookie refresh for Safari
                if (this.isWebKit()) {
                    this.setupCookieRefresh(token);
                }
                return true;
            }
            return false;
        } catch (e) {
            console.error('Error setting cookie:', e);
            return false;
        }
    },

    isWebKit: function() {
        return /WebKit/.test(navigator.userAgent) && 
               !/Chrome/.test(navigator.userAgent);
    },
    
    setupCookieRefresh: function(token) {
        // Try to refresh the cookie every 12 hours to prevent ITP expiration
        const TWELVE_HOURS = 12 * 60 * 60 * 1000;
        
        const refresh = () => {
            const currentCookie = this.getAccessCookie();
            if (currentCookie) {
                try {
                    const decodedToken = JSON.parse(atob(currentCookie));
                    const expiryTimestamp = decodedToken.expires * 1000;
                    
                    // Only refresh if the cookie hasn't actually expired
                    if (expiryTimestamp > Date.now()) {
                        this.setAccessCookie(currentCookie);
                        console.log('Refreshed cookie to prevent ITP expiration');
                    }
                } catch (e) {
                    console.error('Error refreshing cookie:', e);
                }
            }
        };

        // Set up periodic refresh
        setInterval(refresh, TWELVE_HOURS);
        
        // Also refresh on visibility change
        document.addEventListener('visibilitychange', () => {
            if (document.visibilityState === 'visible') {
                refresh();
            }
        });
    },

    getVoucherState: function() {
        const cookie = this.getAccessCookie();
        if (!cookie) return null;
        
        try {
            const decodedToken = JSON.parse(atob(cookie));
            // Check if the token is still valid according to its internal expiration
            if (decodedToken.expires * 1000 > Date.now()) {
                return decodedToken;
            }
        } catch (e) {
            console.error('Error parsing voucher state:', e);
        }
        return null;
    },

    getAccessCookie: function() {
        const name = this.config.cookieName + "=";
        const decodedCookie = decodeURIComponent(document.cookie);
        const cookieArray = decodedCookie.split(';');
        
        for (let i = 0; i < cookieArray.length; i++) {
            let cookie = cookieArray[i].trim();
            if (cookie.indexOf(name) === 0) {
                return cookie.substring(name.length, cookie.length);
            }
        }
        return null;
    },

    getBaseDomain: function(hostname) {
        const parts = hostname.split('.');
        if (parts.length > 2) {
            return parts.slice(-2).join('.');
        }
        return hostname;
    },

    isAllowedOrigin: function(origin) {
        const allowedOrigins = [
            'https://devmapa.nabezky.sk',
            'https://mapa.nabezky.sk'
        ];
        return allowedOrigins.includes(origin);
    }
};